#! /usr/bin/env perl

# generate_css_data.pl: generate perl hashes and C structures containing
# CSS data for HTML conversion.
#
# Copyright 2024-2026 Free Software Foundation, Inc.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License,
# or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <https://www.gnu.org/licenses/>.

# ../maintain/generate_css_data.pl ../data/default_css_element_class_styles.csv perl Texinfo/HTMLDataCSS.pm

use strict;

use warnings;

# in Perl core standard modules
use File::Basename;
use Text::Wrap;

my $program_name = basename($0);

my $base_default_css_element_class_styles_file = $ARGV[0];

open (BDCSS, "<$base_default_css_element_class_styles_file")
  or die "open $base_default_css_element_class_styles_file failed: $!";


my @header;
while (1) {
  my $header_line = <BDCSS>;
  chomp($header_line);
  #print STDERR "$header_line\n";
  @header = split(/\|/, $header_line);
  if (scalar(@header) > 1) {
    last;
  }
}

my ($selector_index, $style_index, $notes_index);
my $header_index = 0;
foreach my $header (@header) {
  if ($header eq 'selector') {
    $selector_index = $header_index;
  } elsif ($header eq 'style') {
    $style_index = $header_index;
  } elsif ($header eq 'notes') {
    $notes_index = $header_index;
  }
  $header_index++;
}
if (!defined($selector_index) or !defined($style_index)
    or !defined($notes_index)) {
  die "missing header column ($selector_index, $style_index, $notes_index)\n";
}

my $lang = $ARGV[1];

my $perl_format = 0;
my $C_format = 0;
if ($lang eq 'perl') {
  $perl_format = 1;
} else {
  $C_format = 1;
}

my $out_file = $ARGV[2];

die "Need an output file\n" if (!defined($out_file));

open(OUT, ">$out_file") or die "Open $out_file: $!\n";

my $initial_notes_tab;
my $subsequent_notes_tab;

my $base_default_css_element_class_styles_base_name
  = basename($base_default_css_element_class_styles_file);
if ($perl_format) {
  $initial_notes_tab = ' # ';
  $subsequent_notes_tab = ' # ';

  print OUT "# Automatically generated by $program_name\n\n";

  print OUT "package Texinfo::HTMLDataCSS;\n\n";

  print OUT "use Texinfo::Common;\n\n";

  print OUT "# Generated from $base_default_css_element_class_styles_base_name\n";
  print OUT "my %base_default_css_element_class_styles = (\n";

} else {
  # C format

  my $header_file = $ARGV[3];

  die "Need an output header\n" if (!defined($header_file));

  open(HDR, ">$header_file") or die "Open $header_file: $!\n";

  $initial_notes_tab = '';
  $subsequent_notes_tab = '   ';

  print OUT "/* Automatically generated by $program_name */\n\n";

  print OUT "#include <config.h>\n\n";
  print OUT "#include \"converter_types.h\"\n\n";

  print OUT "/* Generated from $base_default_css_element_class_styles_base_name */\n";
  print OUT "const CSS_SELECTOR_STYLE base_default_css_element_class_styles[] = {\n";
}

my $css_element_class_style_nr = 0;
while (<BDCSS>) {
  chomp;
  my @data = split (/\|/);
  my $notes = $data[$notes_index];
  if (defined($notes) and $notes ne '') {
    my $lines;
    if ($perl_format) {
      $lines = wrap($initial_notes_tab, $subsequent_notes_tab, ($notes));
      print OUT $lines."\n";
    } else {
      $lines = wrap($initial_notes_tab, $subsequent_notes_tab, ($notes . ' */'));
      print OUT "/* $lines\n";
    }
  }
  my $selector = $data[$selector_index];
  if (!defined($selector) or $selector eq '') {
    die "$base_default_css_element_class_styles_file: Bad selector\n";
  }
  my $style = $data[$style_index];
  if (!defined($style) or $style eq '') {
    die "$base_default_css_element_class_styles_file: Bad style\n";
  }
  $css_element_class_style_nr++;
  if ($perl_format) {
    print OUT "    '$selector'    => '$style',\n";
  } else {
    print OUT "    {\"$selector\", \"$style\"},\n";
  }
}

my $C_header_string = 'HTML_CSS_DATA_H';
if ($perl_format) {

  print OUT ");\n\n";

  print OUT 'sub get_base_default_css_info() {'."\n"
  .'  return \%base_default_css_element_class_styles;'."\n"
  ."}\n\n";
} else {

  print OUT "    {0, 0}\n";
  print OUT "};\n\n";

  print HDR "/* Automatically generated from $program_name */\n\n";
  print HDR "#ifndef $C_header_string\n"
         ."#define $C_header_string\n\n";

  print HDR "#define BASE_DEFAULT_CSS_ELEMENT_CLASS_STYLE_NR "
               ."$css_element_class_style_nr\n\n";
  print HDR "#endif\n";
}




close(OUT);
